﻿using System;
using System.Runtime.InteropServices;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Windows.Forms;
using System.Globalization;
using Microsoft.Office.Interop.Visio;
using System.Threading.Tasks;
using VisOcx = AxMicrosoft.Office.Interop.VisOcx;

#region Особенности плагина импорта Visio
/*
 * 1. Константа PinConnectionGuid отвечает за хранение ссылки на объект, на котором лежит текущая фигура.
 * Если его заполнить, BusinessStudio прикрепит одну фигуру к другой и автоматически создаст связь, если она возможна.
 * Константа доступна только на графическом этапе импорта, так как было решено, что создать связь не на основе фигуры неправильно.
 *
 * 2. Особенности стрелок.
 * Константы ShapeFrom_guid и ShapeTo_guid отвечают за хранение гуидов конечных фигур, не зависит от того стрелка на конце или 2D фигура.
 * Константы EndArrow и BeginArrow отвечают за хранение типа начала и конца стрелки. Хранятся в Visio как цифры, могут однозначно интерпретироваться с помощью встроенных инструментов в импорте  BusinessStudio.
 * 
 * 3. Некоторые константы были продублированы, например ShapeFrom_guid и From_guid. Это было необходимо для того, чтобы одновременно иметь гуид, полученный из Visio(константы с приставкой Shape)
 * и гуид, который объект имеет в BusinessStudio(константы, хранящие гуид и не имеющие приставку Shape переписываются на этапе графического импора, если ключом импорта был не гуид).
 * 
 * 4. Константа UnicodeObject содержит константу, которая может хранится в тексте некоторых фигур.
 * Поэтому при получении текста, нужно исключить возможность попадания этой константы в поле текста.
 * 
 * 5. Координаты центра каждой фигуры хранятся в константах XCenter и YCenter. 
 * У стрелок дополнительно хранятся точки узлов геометрии в константе Geometry. Хранятся в виде пар x1;y1;x2;y2 и т.д.
 * Все координаты в дюймах, так как визио использует в некоторых методах безразмерные double величины, которые считаются дюймами.
 * Все размеры и координаты должны быть в дюймах, иначе их размеры и/или положение не будут совпадать с реальными.
 * 
 * 6. Нельзя менять или убирать параметры из методов, ответственных за графичекий импорт(GetGraphicData, GetGraphicParameters и все методы, что они вызывают), они должны быть все.
 * Параметры объектного импорта можно изменять и/или убирать. Необходимая константа Diagram, определяет объекты, считающиеся диаграммой, используется в объектном и в графическом этапах импорта.
 * 
 * 7. Всем параметрам из секции Prop дописывается в название константа Prop, аналогично с параметрами секции User, так как в секциях User и Prop могут быть праметры с одинаковым названием.
 *  
 * 
*/
#endregion

namespace Plugin//10340
{
    public class Import
    {
        /// <summary>
        /// Хранит массив путей к выбранным файлам.
        /// </summary>
        private static string[] FilesPath;

        /// <summary>
        /// Приложение с помощью которого происходит взаимодействие с Visio.
        /// </summary>
        private static Microsoft.Office.Interop.Visio.Application VisioApplication;

        /// <summary>
        /// Название параметра, являющегося идентификатором того, что текущий класс - диаграмма.
        /// </summary>
        private const string Diagram = "Diagram";

        #region Предопределенные параметры в этапах импорта.
        /// <summary>
        /// Название параметра хранящего глобальный идентификатор объекта на который наведена ссылка.
        /// </summary>
        private const string BeginArrow = "BeginArrow";
        /// <summary>
        /// Название параметра хранящего глобальный идентификатор объекта на который наведена ссылка.
        /// </summary>
        private const string EndArrow = "EndArrow";
        /// <summary>
        /// Название параметра хранящего глобальный идентификатор объекта на который наведена ссылка.
        /// </summary>
        private const string Hyperlink_guid = "Hyperlink_guid";
        /// <summary>
        /// Название параметра хранящего глобальный идентификатор объекта на который наведена ссылка.
        /// </summary>
        private const string ShapeHyperlink_guid = "ShapeHyperlink_guid";
        /// <summary>
        /// Название параметра хранящего текст ссылки.
        /// </summary>
        private const string Hyperlink = "Hyperlink";
        /// <summary>
        /// Название параметра хранящего ссылку на объект связанный с текущим в точке.
        /// </summary>
        private const string PinConnectionGuid = "PinConnectionGuid";
        /// <summary>
        /// Название параметра хранящего имя мастера контейнера в котором лежит текущий объект.
        /// </summary>
        private const string ContainerMaster = "ContainerMaster";
        /// <summary>
        /// Название параметра хранящего имя мастера контейнера хранящего контейнер в котором лежит текущий объект.
        /// </summary>
        private const string ContainerContainerMaster = "ContainerContainerMaster";
        /// <summary>
        /// Название параметра хранящего размер шрифта фигуры.
        /// </summary>
        private const string FontSize = "FontSize";
        /// <summary>
        /// Название параметра хранящего ссылку на фигуру контейнер в котором лежит фигура.
        /// </summary>
        private const string Container_guid = "Container_guid";
        /// <summary>
        /// Название параметра хранящего ссылку на фигуру контейнер в котором лежит фигура.
        /// </summary>
        private const string ShapeContainer_guid = "ShapeContainer_guid";
        /// <summary>
        /// Название параметра хранящего глобальный идентификатор фигуры. Для графического этапа импорта.
        /// </summary>
        private const string Shape_guid = "Shape_guid";
        /// <summary>
        /// Название параметра хранящего глобальный идентификатор фигуры. Для объектного этапа импорта.
        /// </summary>
        private const string guid = "guid";
        /// <summary>
        /// Название параметра хранящего глобальный идентификатор родителя фигуры. Считаем родителем диаграмму на которой размещена фигура.
        /// </summary>
        private const string ParentGuid = "ParentGuid";
        /// <summary>
        /// Название параметра хранящего глобальный идентификатор родителя фигуры. Считаем родителем диаграмму на которой размещена фигура.
        /// </summary>
        private const string ShapeParentGuid = "ShapeParentGuid";
        /// <summary>
        /// Название параметра хранящего текст фигуры.
        /// </summary>
        private const string Text = "Text";
        /// <summary>
        /// Название параметра хранящего название мастера фигуры.
        /// </summary>
        private const string Master = "Master";
        /// <summary>
        /// Название параметра хранящего ширину фигуры.
        /// </summary>
        private const string Width = "Width";
        /// <summary>
        /// Название параметра хранящего высоту фигуры.
        /// </summary>
        private const string Height = "Height";
        /// <summary>
        /// Название параметра хранящего координату X центра фигуры.
        /// </summary>
        private const string XCenter = "XCenter";
        /// <summary>
        /// Название параметра хранящего координату Y центра фигуры.
        /// </summary>
        private const string YCenter = "YCenter";
        /// <summary>
        /// Название параметра хранящего координату X левогй грани фигуры.
        /// </summary>
        private const string XLeft = "XLeft";
        /// <summary>
        /// Название параметра хранящего координату Y верхней грани фигуры.
        /// </summary>
        private const string YTop = "YTop";
        /// <summary>
        /// Название параметра хранящего координату X левогй грани фигуры.
        /// </summary>
        private const string XRight = "XRight";
        /// <summary>
        /// Название параметра хранящего координату Y верхней грани фигуры.
        /// </summary>
        private const string YBottom = "YBottom";
        #endregion

        #region Параметры страницы
        /// <summary>
        /// Название параметра графического этапа, хранящего ширину страницы.
        /// </summary>
        private const string PageWidth = "PageWidth";
        /// <summary>
        /// Название параметра графического этапа, хранящего высоту страницы.
        /// </summary>
        private const string PageHeight = "PageHeight";
        /// <summary>
        /// Название параметра объектного этапа, хранящего текст из фигуры-контейнера. Считаем одни из вариантов хранения имени диаграммы.
        /// </summary>
        private const string ContainerText = "ContainerText";
        /// <summary>
        /// Название параметра объектного этапа, хранящего имя файла. Считаем одни из вариантов хранения имени диаграммы.
        /// </summary>
        private const string FileName = "FileName";
        #endregion

        #region Предопределенные параметры , соответвующие ID фигур Visio, заполняются  в BS после графического импорта.
        /// <summary>
        /// Название опции хранящей идентификатор фигуры в BS. Хранится идентификатор фигуры из которой выходит стрелка. Будет заполнен вне плагина.
        /// </summary>
        private const string ShapeFrom_id = "ShapeFrom_id";
        /// <summary>
        /// Название опции хранящей идентификатор фигуры в BS. Хранится идентификатор фигуры в которую входит стрелка. Будет заполнен вне плагина.
        /// </summary>
        private const string ShapeTo_id = "ShapeTo_id";
        /// <summary>
        /// Название опции хранящей идентификатор фигуры в BS. Будет заполнен вне плагина.
        /// </summary>
        private const string Shape_id = "Shape_id";
        #endregion

        #region Параметры стрелок
        /// <summary>
        /// Название параметра содержащего строку координат вида x1;y1;x2;y2
        /// </summary>
        private const string Geometry = "Geometry";
        /// <summary>
        /// Шаблон для дополнения мастера стрелки мастером фигуры из которой выходит стрелка.
        /// </summary>
        private const string TitleFrom = "_From_{0}";
        /// <summary>
        /// Шаблон для дополнения мастера стрелки мастером фигуры в которую входит стрелка.
        /// </summary>
        private const string TitleTo = "_To_{0}";
        /// <summary>
        /// Название параметра содержащего guid фигуры из которой выходит стрелка.
        /// </summary>
        private const string ShapeFrom_guid = "ShapeFrom_guid";
        /// <summary>
        /// Название параметра содержащего guid фигуры из которой выходит стрелка.
        /// </summary>
        private const string From_guid = "From_guid";
        /// <summary>
        /// Название параметра содержащего guid фигуры в которую входит стрелка.
        /// </summary>
        private const string ShapeTo_guid = "ShapeTo_guid";
        /// <summary>
        /// Название параметра содержащего guid фигуры в которую входит стрелка.
        /// </summary>
        private const string To_guid = "To_guid";
        /// <summary>
        /// Название параметра содержащего координату X начала стрелки.
        /// </summary>
        private const string BeginX = "BeginX";
        /// <summary>
        /// Название параметра содержащего координату Y начала стрелки.
        /// </summary>
        private const string BeginY = "BeginY";
        /// <summary>
        /// Название параметра содержащего координату X конца стрелки.
        /// </summary>
        private const string EndX = "EndX";
        /// <summary>
        /// Название параметра содержащего координату Y конца стрелки.
        /// </summary>
        private const string EndY = "EndY";
        #endregion

        #region System constants
        /// <summary>
        /// Константа обозначает символ, хранящийся в некоторых фигурах Visio в свойстве Text.
        /// </summary>
        private const char UnicodeObject = '\uFFFC';
        /// <summary>
        /// Константа предназначена для маркировки повторяющихся названий у строк.
        /// </summary>
        private const string Properties = "Prop";
        /// <summary>
        /// Константа предназначена для маркировки повторяющихся названий у строк.
        /// </summary>
        private const string UserProperties = "User";
        /// <summary>
        /// Фильтр, необходим для того, чтобы выбирать файлы требуемого расширения.
        /// </summary>
        private const string Filter = "Файлы Visio|*.vsd?";
        /// <summary>
        /// Заголовок окна с выбором файлов.
        /// </summary>
        private const string TitleName = "Укажите файлы Visio для импорта";
        /// <summary>
        /// Текст ошибки при выборе файла с неподдерживаемым расширением.
        /// </summary>
        private const string ExtensionError = "Выбранный файл {0} имеет некорректное расширение {1}";
        #endregion

        #region Hashtables
        /// <summary>
        /// Таблица хранит пару глобальный идентификатор фигуры - все данные для объектного этапа импорта этой фигуры.
        /// Требуется для того, чтобы на графическом этапе импорта сразу обращаться к данным, а не формировать заново.
        /// </summary>
        private static Hashtable ShapeDataList = new Hashtable();
        /// <summary>
        /// Таблица хранит пару глобальный идентификатор фигуры - все данные для графического этапа импорта этой фигуры.
        /// </summary>
        private static Hashtable GraphicShapeDataList = new Hashtable();
        /// <summary>
        /// Таблица хранит пару имя класса - все параметры для графического этапа импорта этого класса.
        /// Требуется для того, чтобы на графическом этапе импорта сразу обращаться к параметрам, а не формировать заново.
        /// </summary>
        private static Hashtable GraphicParameterTable = new Hashtable();
        /// <summary>
        /// Таблица хранит пару имя класса - все параметры для объектного этапа импорта этого класса.
        /// Требуется для того, чтобы на объектном этапе импорта сразу обращаться к параметрам, а не формировать заново.
        /// </summary>
        private static Hashtable ObjectParameterTable = new Hashtable();
        /// <summary>
        /// Таблица хранит пару глобальный идентификатор фигуры - мастер фигуры.
        /// </summary>
        private static Hashtable MastersTable = new Hashtable();
        #endregion

        /// <summary>
        /// Возвращаемая строка будет отображаться в списке доступных плагинов для формирования пакета импорта.
        /// </summary>
        /// <returns>Возвращает строку с названием плагина</returns>
        public static string GetPluginTitle()
        {
            return "Visio Import Plugin v1.00";
        }
        /// <summary>
        /// Метод открытия плагина. Запоминает выбранные файлы. Создаёт новое приложение.
        /// </summary>
        /// <returns>True, если всё прошло без ошибок</returns>
        public static bool Open()
        {
            try
            {
                using (OpenFileDialog ofd = new OpenFileDialog())
                {
                    ofd.Title = TitleName;
                    ofd.Multiselect = true;
                    ofd.Filter = Filter;
                    //Диалог выбора файлов.
                    if (ofd.ShowDialog() == DialogResult.OK)
                    {
                        //Проверка, все ли выбранные файлы поддерживаемого расширения
                        foreach (string path in ofd.FileNames)
                        {
                            if (!System.IO.Path.GetExtension(path).ToLower().Equals(".vsd") && !System.IO.Path.GetExtension(path).ToLower().Equals(".vsdx"))
                                throw new Exception(String.Format(ExtensionError, path, System.IO.Path.GetExtension(path)));
                        }
                        //Заполняем глобальные переменные и создаём форму импорта
                        FilesPath = ofd.FileNames;
                        VisioApplication = new Microsoft.Office.Interop.Visio.Application();
                        VisioApplication.Visible = false;
                        
                        return true;
                    }
                    else
                        return false;
                }
            }
            catch (Exception ex)
            {
                throw new Exception(ex.Message, ex.InnerException);
            }
        }

        /// <summary>
        /// Определяет количество выбранных информационных пакетов.
        /// </summary>
        /// <returns>Количество импортируемых информационных пакетов</returns>
        public static int GetPartsCount()
        {
            return FilesPath.Length;
        }

        /// <summary>
        /// Текущий открытый документ.
        /// </summary>
        private static Document VisioDoc = null;

        /// <summary>
        /// Получает документ по его номеру.
        /// </summary>
        /// <param name="partNumber">Номер текущего информационного пакета</param>
        /// <returns>Возвращает документ с требуемым именем</returns>
        private static Document GetDocumentByPartNumber(int partNumber)
        {
            string path = FilesPath[partNumber];

            if (partNumber > 0)
            {
                if (!VisioDoc.Name.Equals(System.IO.Path.GetFileName(path)))
                {
                    VisioDoc.Close();
                    MastersTable.Clear();
                    GraphicShapeDataList.Clear();
                    ShapeDataList.Clear();
                    ObjectParameterTable.Clear();
                    GraphicParameterTable.Clear();

                    VisioDoc = VisioApplication.Documents.OpenEx(path, (short)VisOpenSaveArgs.visOpenRO);
                    //Обновляем все фигуры, чтобы у них был статичный глобальный идентификатор
                    UpdateAllShapes(VisioDoc);
                    //Сохраняем. Теперь глобальный идентификатор не будет меняться
                    VisioDoc.SaveAs(path);

                    return VisioDoc;
                }
                else return VisioDoc;
            }
            else
            {
                if (VisioDoc != null) return VisioDoc;
                else
                {
                    VisioDoc = VisioApplication.Documents.OpenEx(path, (short)VisOpenSaveArgs.visOpenRO);
                    //Обновляем все фигуры, чтобы у них был статичный глобальный идентификатор
                    UpdateAllShapes(VisioDoc);
                    //Сохраняем. Теперь глобальный идентификатор не будет меняться
                    VisioDoc.SaveAs(path);

                    return VisioDoc;
                }
            }
        }

        /// <summary>
        /// Метод создаёт глобальные идентификаторы всем фигурам на всех страницах файла, затем сохраняет документ.
        /// Это позволяет присвоить эти идентификаторы фигурам, при последующих вызовах они не будут меняться.
        /// </summary>
        /// <param name="doc">Текущий документ</param>
        private static void UpdateAllShapes(Document doc)
        {
            foreach (Page page in doc.Pages)
            {
                GetShapeGUID(page.PageSheet);
                foreach (Shape shape in page.Shapes)
                    GetShapeGUID(shape);
            }
        }

        /// <summary>
        /// Обязательный метод плагина импорта. Получает список всех классов(мастеров), используемых на диаграммах.
        /// </summary>
        /// <param name="partNumber">Номер текущего информационного пакета</param>
        /// <returns>Возвращает список всех мастеров</returns>
        public static List<string> GetClasses(int partNumber)
        {
            try
            {
                List<string> classes = new List<string>();

                {
                    Document doc = GetDocumentByPartNumber(partNumber);
                    foreach (Page page in doc.Pages)
                        foreach (Shape shape in page.Shapes)
                        {
                            //Если фигура - стрелка, нужно забрать дополнительно общий класс
                            if (IsOneD(shape))
                                AddWithCheck(classes, GetMasterName(shape, true));
                            AddWithCheck(classes, GetMasterName(shape));
                        }
                }
                AddWithCheck(classes, Diagram);
                return classes;
            }
            catch (Exception ex)
            {
                throw new Exception(ex.Message, ex.InnerException);
            }
        }

        /// <summary>
        /// Определяет является ли фигура стрелкой.
        /// </summary>
        /// <param name="shape">Определяемая фигура</param>
        /// <returns></returns>
        private static bool IsOneD(Shape shape)
        {
            return shape.OneD != 0;
        }

        /// <summary>
        /// Обазательный класс. Получает список всех графических классов.
        /// </summary>
        /// <param name="partNumber">Номер текущего информационного пакета</param>
        /// <returns>Возвращает графические классы объектов</returns>
        public static List<string> GetGraphicClasses(int partNumber)
        {
            try
            {
                List<string> classes = new List<string>();

                {
                    Document doc = GetDocumentByPartNumber(partNumber);

                    foreach (Page page in doc.Pages)
                        foreach (Shape shape in page.Shapes)
                            AddWithCheck(classes, GetMasterName(shape, true));
                }

                return classes;
            }
            catch (Exception ex)
            {
                throw new Exception(ex.Message, ex.InnerException);
            }
        }

        /// <summary>
        /// Обязательный метод плагина импорта. Получает параметры фигур принадлежащие определённому мастеру.
        /// </summary>
        /// <param name="className">Имя мастера. Получаем заголовки параметров у объектов данного мастера</param>
        /// <param name="partNumber">Номер текущего информационного пакета</param>
        /// <returns>Возвращает параметры фигур принадлежащие определённому мастеру</returns>
        public static List<string> GetParameters(string className, int partNumber)
        {
            try
            {
                if (ObjectParameterTable.Count > 0 && ObjectParameterTable.ContainsKey(className))
                    return (List<string>)ObjectParameterTable[className];
                List<string> parameters = new List<string>();

                Document doc = GetDocumentByPartNumber(partNumber);
                foreach (Page page in doc.Pages)
                {
                    //Обязательные параметры, должны быть у каждой фигуры
                    AddWithCheck(parameters, guid);
                    AddWithCheck(parameters, Shape_guid);
                    AddWithCheck(parameters, ParentGuid);
                    AddWithCheck(parameters, ShapeParentGuid);
                    AddWithCheck(parameters, Text);
                    AddWithCheck(parameters, PinConnectionGuid);

                    if (className.Equals(Diagram))//Для диаграммы из секции User и Prop свойства не забираем
                    {
                        AddWithCheck(parameters, ContainerText);
                        AddWithCheck(parameters, FileName);
                        break;
                    }
                    foreach (Shape shape in page.Shapes)
                    {
                        if (className.Equals(GetMasterName(shape)) || className.Equals(GetMasterName(shape, true)))
                        {
                            //Для стрелок дополнительно нужно добавить заголовки с гуидом фигур из которой стрелка выходит и в которую входит
                            if (IsOneD(shape))
                            {
                                AddWithCheck(parameters, ShapeFrom_guid);
                                AddWithCheck(parameters, ShapeTo_guid);
                                AddWithCheck(parameters, From_guid);
                                AddWithCheck(parameters, To_guid);
                                AddWithCheck(parameters, BeginArrow);
                                AddWithCheck(parameters, EndArrow);
                            }

                            AddWithCheck(parameters, Container_guid);
                            AddWithCheck(parameters, ShapeContainer_guid);
                            AddWithCheck(parameters, ContainerMaster);
                            AddWithCheck(parameters, ContainerContainerMaster);

                            //Добавляем параметры, отвечающие за гиперссылку, даже если отсутстсвует. Это гарантирует корректную работу импорта.
                            AddWithCheck(parameters, Hyperlink);
                            AddWithCheck(parameters, Hyperlink_guid);
                            AddWithCheck(parameters, ShapeHyperlink_guid);

                            short rowIndices = (short)VisRowIndices.visRowFirst;
                            short sectionIndices = (short)VisSectionIndices.visSectionProp;
                            short cellIndices = (short)VisCellIndices.visCustPropsLabel;
                            //Забираем заголовки параметров из секции Prop
                            while (CellExist(shape, sectionIndices, rowIndices, cellIndices))
                            {
                                string propName = string.Concat(Properties, shape.get_CellsSRC(sectionIndices, rowIndices, cellIndices).RowName);
                                AddWithCheck(parameters, propName);

                                rowIndices++;
                            }
                            rowIndices = (short)VisRowIndices.visRowFirst;
                            sectionIndices = (short)VisSectionIndices.visSectionUser;
                            cellIndices = (short)VisCellIndices.visUserValue;
                            //Забираем заголовки параметров из секции User
                            while (CellExist(shape, sectionIndices, rowIndices, cellIndices))
                            {
                                string propName = string.Concat(UserProperties, shape.get_CellsSRC(sectionIndices, rowIndices, cellIndices).RowName);
                                AddWithCheck(parameters, propName);

                                rowIndices++;
                            }

                            break;
                        }
                    }
                }
                //Обязательные параметры, должны быть у каждой фигуры
                //Заполняются пустыми строками в методе AddFilledUpDataList, требуется соблюдать порядок добавления/заполнения в обоих методах.
                AddWithCheck(parameters, Shape_id);
                AddWithCheck(parameters, ShapeFrom_id);
                AddWithCheck(parameters, ShapeTo_id);
                ObjectParameterTable.Add(className, parameters);
                return parameters;
            }
            catch (Exception ex)
            {
                throw new Exception(ex.Message, ex.InnerException);
            }
        }

        /// <summary>
        /// Получает контейнер содержащий текущую фигуру.
        /// </summary>
        /// <param name="page">Страница у которой ищем шейпы входящие в контейнер</param>
        /// <param name="containingShape">Фигура лежащая в контейнере</param>
        /// <returns>Нижний контейнер, содержащий фигуру</returns>
        private static Shape GetCurrentContainer(Page page, Shape containingShape)
        {
            try
            {
                if (containingShape == null) return null;
                //Получаем все контейнеры которые содержат текущую фигуру
                int[] containers = (int[])containingShape.MemberOfContainers;
                if (containers == null || containers.Length == 0) return null;

                if (containers.Length > 1)
                {
                    List<double> areas = new List<double>();
                    for (int i = 0; i < containers.Length; i++)
                    {
                        Shape shape = page.Shapes.get_ItemFromID(containers[i]);
                        //Если тип контейнера не группа, не будем его учитывать
                        if (shape.Type != (short)VisShapeTypes.visTypeGroup)
                        {
                            areas.Add(double.MaxValue);
                            continue;
                        }
                        //Добавляем площадь контейнера, предположим, что искомая площадь минимальная
                        areas.Add(shape.Cells[Width].ResultIU * shape.Cells[Height].ResultIU);
                    }
                    return page.Shapes.get_ItemFromID(containers[areas.FindIndex(val => val.Equals(areas.Min()))]);
                }
                else if (containers.Length == 1)
                {
                    Shape shape = page.Shapes.get_ItemFromID(containers[0]);
                    if (shape.Type != (short)VisShapeTypes.visTypeGroup)
                        shape = GetCurrentContainer(page, shape);
                    if (shape != null && shape.Type == (short)VisShapeTypes.visTypeGroup)
                        return shape;
                }

                return null;
            }
            catch (Exception ex)
            {
                throw new Exception(ex.Message, ex.InnerException);
            }
        }

        /// <summary>
        /// Получаем самый верхний контейнер из лежащих на диаграмме.
        /// </summary>
        /// <param name="page">Страница у которой ищем контейнер</param>
        /// <param name="containers">Список идентификаторов всех контейнеров</param>
        /// <returns>Возвращает самый верхний контейнер</returns>
        private static Shape GetTopContainer(Page page, int[] containers)
        {
            try
            {
                for (int i = 0; i < containers.Length; i++)
                {
                    Shape shp = page.Shapes.get_ItemFromID(containers[i]);
                    if (shp.Type != (short)VisShapeTypes.visTypeGroup) continue;
                    if (shp.MemberOfContainers.Length > 0)
                    {
                        foreach (int j in shp.MemberOfContainers)
                            if (Array.IndexOf(containers, j) == -1)
                            {
                                Array.Resize(ref containers, containers.Length + 1);
                                containers[containers.Length - 1] = j;
                            }

                        continue;
                    }
                    return shp;
                }
                return null;
            }
            catch (Exception ex)
            {
                throw new Exception(ex.Message, ex.InnerException);
            }
        }

        /// <summary>
        /// Добавляет значение к списку с проверкой.
        /// </summary>
        /// <param name="listValues">Список к которому добавляем значение</param>
        /// <param name="value">Добавляемое значение</param>
        private static void AddWithCheck(List<string> listValues, string value)
        {
            if (!listValues.Exists(str => str.Equals(value)))
                listValues.Add(value);
        }

        /// <summary>
        /// Получение имени мастера фигуры. 
        /// Для стрелок дополнительно получаем мастера фигуры из которой выходит стрелка и в которую входит.
        /// </summary>
        /// <param name="shape">Фигура у которой получаем мастера</param>
        /// <param name="isOneDFromTo">Опция предназначенная для получения мастера без учёта особенностей стрелок</param>
        /// <returns>Возвращает обработанное имя мастера</returns>
        private static string GetMasterName(Shape shape, bool isOneDFromTo = false)
        {
            try
            {
                if (MastersTable.Count > 0 && MastersTable.ContainsKey(GetShapeGUID(shape)))
                {
                    string master = MastersTable[GetShapeGUID(shape)].ToString();
                    //Если фигура - стрелка и включена опция, отрезаем у мастера стрелки From To если есть
                    if (IsOneD(shape) && isOneDFromTo)
                    {
                        int titleFrom = master.IndexOf(TitleFrom.Remove(TitleFrom.IndexOf('{')));
                        if (titleFrom != -1)
                            master = master.Remove(titleFrom);

                        int titleTo = master.IndexOf(TitleTo.Remove(TitleTo.IndexOf('{')));
                        if (titleTo != -1)
                            master = master.Remove(titleTo);
                        return master;
                    }

                    return master;
                }
                //Отрезаем ненужные части от мастера
                string masterName;
                if (shape.Master != null)
                    masterName = shape.Master.NameU.LastIndexOf('.') > 0 ? shape.Master.NameU.Substring(0, shape.Master.NameU.LastIndexOf('.')) : shape.Master.NameU;
                else//Если мастера нет, то считаем именем мастера свойство NameU
                    masterName = shape.NameU.LastIndexOf('.') > 0 ? shape.NameU.Substring(0, shape.NameU.LastIndexOf('.')) : shape.NameU;

                if (IsOneD(shape) && !isOneDFromTo)
                {
                    //Для стрелок получаем мастера вместе с мастерами фигур из которой выходит в которую входит стрелка
                    string fromCon, toCon;
                    fromCon = toCon = String.Empty;
                    if (shape.Connects != null && shape.Connects.Count > 0)
                        foreach (Connect connect in shape.Connects)
                        {
                            if (connect.FromPart.Equals((short)VisFromParts.visBegin))
                                fromCon = String.Format(TitleFrom, GetMasterName(connect.ToSheet, true));

                            else if (connect.FromPart.Equals((short)VisFromParts.visEnd))
                                toCon = String.Format(TitleTo, GetMasterName(connect.ToSheet, true));
                        }

                    masterName = masterName + fromCon + toCon;
                }

                MastersTable.Add(GetShapeGUID(shape), masterName);

                return masterName;
            }
            catch (Exception ex)
            {
                throw new Exception(ex.Message, ex.InnerException);
            }
        }

        /// <summary>
        /// Получает данные находящиеся в ячейке.
        /// </summary>
        /// <param name="shape">Фигура из которой извлекаем данные</param>
        /// <param name="section">Константа(short) отвечающая за определённую секцию</param>
        /// <param name="row">Константа(short) отвечающая за определённую строку</param>
        /// <param name="cell">Константа(short) отвечающая за определённую ячейку</param>
        /// <returns>Возвращает значение находящееся в ячейке</returns>
        private static string GetCellValue(Shape shape, short section, short row, short cell, VisUnitCodes option = VisUnitCodes.visNoCast)
        {
            if (CellExist(shape, section, row, cell))
                return shape.get_CellsSRC(section, row, cell).get_ResultStrU(option);
            else
                return String.Empty;
        }

        /// <summary>
        /// Проверяет наличие ячейки в фигуре.
        /// </summary>
        /// <param name="shape">Фигура в которой ищем секцию</param>
        /// <param name="section">Константа(short) отвечающая за определённую секцию</param>
        /// <param name="row">Константа(short) отвечающая за определённую строку</param>
        /// <param name="cell">Константа(short) отвечающая за определённую ячейку</param>
        /// <returns>True, когда ячейка присутствует в фигуре</returns>
        private static bool CellExist(Shape shape, short section, short row, short cell)
        {
            return shape.get_CellsSRCExists(section, row, cell, 0) != 0;
        }

        /// <summary>
        /// Обязательный метод пакета импорта. Получает список значений параметров от фигур заданного мастера.
        /// </summary>
        /// <param name="className">Имя мастера. Получаем значения параметров у объектов данного мастера</param>
        /// <param name="partNumber">Номер текущего информационного пакета</param>
        /// <returns>Возвращает список значений параметров от фигур заданного мастера</returns>
        public static List<string[]> GetData(string className, int partNumber)
        {
            try
            {
                List<string[]> master = new List<string[]>();

                string path = FilesPath[partNumber];

                Document doc = GetDocumentByPartNumber(partNumber);
                List<string> parameters = GetParameters(className, partNumber);
                foreach (Page page in doc.Pages)
                {
                    if (className.Equals(Diagram))
                    {
                        AddFilledUpDataList(master, page.PageSheet, page, path, parameters);
                        continue;
                    }
                    foreach (Shape shape in page.Shapes)
                    {
                        if (className.Equals(GetMasterName(shape)) || className.Equals(GetMasterName(shape, true)))
                            AddFilledUpDataList(master, shape, page, path, parameters);
                    }
                }
                //Все пустые значения параметров заполним пустыми строками
                foreach (string[] obj in master)
                    for(int i = 0; i< obj.Length;i++)
                        if (obj[i] == null) obj[i] = string.Empty;

                return master;
            }
            catch (Exception ex)
            {
                throw new Exception(ex.Message, ex.InnerException);
            }
        }

        /// <summary>
        /// Переводит список списков в список массивов.
        /// </summary>
        /// <param name="master">Список списков</param>
        /// <returns>Возвращает список массивов</returns>
        private static List<string[]> ListToArray(List<List<string>> master)
        {
            int lengthBefore = 0;
            int lengthAfter = 1;
            while (lengthBefore != lengthAfter)
            {
                lengthBefore = master.Count;
                master.Remove(master.Find(list => list.Count == 0));
                lengthAfter = master.Count;
            }
            List<string[]> array = new List<string[]>();
            for (int i = 0; i < master.Count; i++)
            {
                array.Add(master[i].ToArray());
            }
            return array;
        }

        /// <summary>
        /// Метод находит номер положения строки в списке.
        /// </summary>
        /// <param name="parameters">Список в котором ищем строку</param>
        /// <param name="parameter">Строка поиска</param>
        /// <returns>Номер положения строки в списке</returns>
        private static int FindArrayIndex(List<string> parameters, string parameter)
        {
            return parameters.FindIndex(str => str.ToUpper().Equals(parameter.ToUpper()));
        }

        /// <summary>
        /// Добавляет заполненный список значений параметров объекта по фигуре к списку объектов.
        /// </summary>
        /// <param name="masterString">Список объектов</param>
        /// <param name="shape">Фигура, у которой забираем объектные данные</param>
        /// <param name="page">Страница для поиска контейнера с возможным именем страницы</param>
        private static void AddFilledUpDataList(List<string[]> masterString, Shape shape, Page page, string path, List<string> parameters)
        {
            string shapeGuid = GetShapeGUID(shape);
            if (ShapeDataList.Count > 0 && ShapeDataList.ContainsKey(shapeGuid))
            {
                masterString.Add((string[])ShapeDataList[shapeGuid]);
                return;
            }

            masterString.Add(new string[parameters.Count]);
            int countString = masterString.Count - 1;
            bool isPage = shape.Type == (short)VisShapeTypes.visTypePage;//Работаем со страницей

            masterString[countString][FindArrayIndex(parameters, guid)] = shapeGuid;
            masterString[countString][FindArrayIndex(parameters, Shape_guid)] = shapeGuid;
            if (!isPage)
            {
                masterString[countString][FindArrayIndex(parameters, ParentGuid)] = GetShapeGUID(page.PageSheet);
                masterString[countString][FindArrayIndex(parameters, ShapeParentGuid)] = GetShapeGUID(page.PageSheet);
                masterString[countString][FindArrayIndex(parameters, Text)] = GetShapeText(shape);
                //Для стрелок получаем глобальные идентификаторы фигур из которой выходит стрелка и в которую входит
                if (IsOneD(shape))
                {
                    string[] connects = GetGuidIn_Out(shape);

                    masterString[countString][FindArrayIndex(parameters, ShapeFrom_guid)] = connects[0];
                    masterString[countString][FindArrayIndex(parameters, ShapeTo_guid)] = connects[1];
                    masterString[countString][FindArrayIndex(parameters, From_guid)] = connects[0];
                    masterString[countString][FindArrayIndex(parameters, To_guid)] = connects[1];

                    //Забираем конец и начало стрелки
                    masterString[countString][FindArrayIndex(parameters, BeginArrow)] = GetCellValue(shape, (short)VisSectionIndices.visSectionObject, (short)VisRowIndices.visRowLine, (short)VisCellIndices.visLineBeginArrow);
                    masterString[countString][FindArrayIndex(parameters, EndArrow)] = GetCellValue(shape, (short)VisSectionIndices.visSectionObject, (short)VisRowIndices.visRowLine, (short)VisCellIndices.visLineEndArrow);
                }
                else
                {
                    string[] connects = GetGuidIn_Out(shape);
                    masterString[countString][FindArrayIndex(parameters, PinConnectionGuid)] = connects[2];
                }
                GetShapeData(masterString, shape, parameters);
            }
            else
            {
                masterString[countString][FindArrayIndex(parameters, ParentGuid)] = String.Empty;
                masterString[countString][FindArrayIndex(parameters, ShapeParentGuid)] = String.Empty;
                masterString[countString][FindArrayIndex(parameters, Text)] = shape.ContainingPage.Name;
                Shape container = GetTopContainer(page, (int[])page.GetContainers(0));
                masterString[countString][FindArrayIndex(parameters, ContainerText)] = container == null ? String.Empty : GetShapeText(container);
                masterString[countString][FindArrayIndex(parameters, FileName)] = System.IO.Path.GetFileNameWithoutExtension(path);
            }
            //Обязательные параметры, должны быть у каждой фигуры
            //Заполняются пустыми строками. В методе GetParameters добавляются заголовки, требуется соблюдать порядок добавления/заполнения в обоих методах.
            masterString[countString][FindArrayIndex(parameters, Shape_id)] = String.Empty;//Заполнение параметра Shape_id
            masterString[countString][FindArrayIndex(parameters, ShapeFrom_id)] = String.Empty;//Заполнение параметра ShapeFrom_id
            masterString[countString][FindArrayIndex(parameters, ShapeTo_id)] = String.Empty;//Заполнение параметра ShapeTo_id

            ShapeDataList.Add(shapeGuid, masterString[countString]);
        }

        /// <summary>
        /// Метод забирает общие данные из фигуры и записывает в список.
        /// </summary>
        /// <param name="masterString">Список, куда записываем данные</param>
        /// <param name="shape">Фигура, у которого считываем данные</param>
        private static void GetShapeData(List<string[]> masterString, Shape shape, List<string> parameters)
        {
            //Для каждой фигуры получаем ссылку на контейнер который содержит эту фигуру, если существует
            Shape container = GetCurrentContainer(shape.ContainingPage, shape);
            string containerGuid = container == null ? String.Empty : GetShapeGUID(container);
            masterString[masterString.Count - 1][FindArrayIndex(parameters, Container_guid)] = containerGuid;
            masterString[masterString.Count - 1][FindArrayIndex(parameters, ShapeContainer_guid)] = containerGuid;
            masterString[masterString.Count - 1][FindArrayIndex(parameters, ContainerMaster)] = container == null ? String.Empty : GetMasterName(container);
            container = GetCurrentContainer(shape.ContainingPage, container);
            masterString[masterString.Count - 1][FindArrayIndex(parameters, ContainerContainerMaster)] = container == null ? String.Empty : GetMasterName(container);
            //Для каждой фигуры проверяем наличие секции гиперссылок и если есть забираем ссылку и глобальный идентификатор объекта ссылки

            short sectionIndices = (short)VisSectionIndices.visSectionHyperlink;
            short rowIndices = (short)VisRowIndices.visRowFirst;

            FillHyperLinkParameterValue(masterString, shape, parameters, sectionIndices, rowIndices);

            //rowIndices = (short)VisRowIndices.visRowFirst;
            sectionIndices = (short)VisSectionIndices.visSectionProp;
            short cell = (short)VisCellIndices.visCustPropsValue;
            //Получаем значения параметров из секции Prop
            while (CellExist(shape, sectionIndices, rowIndices, cell))
            {
                string propName = shape.get_CellsSRC(sectionIndices, rowIndices, cell).RowName;
                if (!String.IsNullOrEmpty(propName))
                {
                    int index = FindArrayIndex(parameters, propName);
                    if (index == -1) index = FindArrayIndex(parameters, string.Concat(Properties, propName));
                    if (index == -1)
                    {
                        rowIndices++;
                        continue;
                    }
                    masterString[masterString.Count - 1][index] = GetCellValue(shape, sectionIndices, rowIndices, cell);
                }
                rowIndices++;
            }

            rowIndices = (short)VisRowIndices.visRowFirst;
            sectionIndices = (short)VisSectionIndices.visSectionUser;
            cell = (short)VisCellIndices.visUserValue;
            //Получаем значения параметров из секции User
            while (CellExist(shape, sectionIndices, rowIndices, cell))
            {
                string propName = shape.get_CellsSRC(sectionIndices, rowIndices, cell).RowName;
                if (!String.IsNullOrEmpty(propName))
                {
                    int index = FindArrayIndex(parameters, propName);
                    if (index == -1) index = FindArrayIndex(parameters, string.Concat(UserProperties, propName));
                    if (index == -1)
                    {
                        rowIndices++;
                        continue;
                    }
                    masterString[masterString.Count - 1][index] = GetCellValue(shape, sectionIndices, rowIndices, cell);
                }
                rowIndices++;
            }
        }

        /// <summary>
        /// Записывает в массив данные секции гиперссылок, если таковая имеется.
        /// </summary>
        /// <param name="masterString">Список, содержащий массивы значений параметров фигур</param>
        /// <param name="shape">Текущая фигура</param>
        /// <param name="parameters">Максимально полный список параметров фигуры текущего класса</param>
        /// <param name="sectionIndices">Число обозначающее текущую секцию фигуры</param>
        /// <param name="rowIndices">Число обозначающее текущую строку в секции фигуры</param>
        private static void FillHyperLinkParameterValue(List<string[]> masterString, Shape shape, List<string> parameters, short sectionIndices, short rowIndices)
        {
            if (shape.get_SectionExists(sectionIndices, 0) != 0)
            {
                string adress = shape.get_CellsSRC(sectionIndices, rowIndices, (short)VisCellIndices.visHLinkAddress).FormulaU;
                string subAdress = shape.get_CellsSRC(sectionIndices, rowIndices, (short)VisCellIndices.visHLinkSubAddress).get_ResultStrU(VisUnitCodes.visNoCast);
                if (String.IsNullOrEmpty(adress) || adress.Equals("\"\""))
                {
                    masterString[masterString.Count - 1][FindArrayIndex(parameters, Hyperlink)] = subAdress;
                    string subAdresGuid = GetHyperlinkGuid(subAdress);
                    masterString[masterString.Count - 1][FindArrayIndex(parameters, Hyperlink_guid)] = subAdresGuid;
                    masterString[masterString.Count - 1][FindArrayIndex(parameters, ShapeHyperlink_guid)] = subAdresGuid;
                }
                else
                {
                    masterString[masterString.Count - 1][FindArrayIndex(parameters, Hyperlink)] = System.IO.Path.GetFileNameWithoutExtension(adress);
                    string adresGuid = GetHyperlinkGuid(adress);
                    masterString[masterString.Count - 1][FindArrayIndex(parameters, Hyperlink_guid)] = adresGuid;
                    masterString[masterString.Count - 1][FindArrayIndex(parameters, ShapeHyperlink_guid)] = adresGuid;
                }
            }
        }
        
        /// <summary>
        /// Получает глобальный идентификатор объекта на которого указывает гиперссылка у объекта.
        /// </summary>
        /// <param name="name">Имя объекта на который указывает ссылка</param>
        /// <returns></returns>
        private static string GetHyperlinkGuid(string name)
        {
            for (int i = 0; i < FilesPath.Length; i++)
            {
                Document doc = GetDocumentByPartNumber(i);
                if (FilesPath[i].Contains(name))
                {
                    List<string> parameters = (List<string>)ObjectParameterTable[Diagram];
                    string[] data = (string[])ShapeDataList[GetShapeGUID(doc.Pages[1].PageSheet)];
                    return data[parameters.FindIndex(str => str.Equals(guid))];
                }
                foreach (Page page in doc.Pages)
                {
                    List<string> parameters = (List<string>)ObjectParameterTable[Diagram];
                    List<string> data = ((string[])ShapeDataList[GetShapeGUID(page.PageSheet)]).ToList();

                    if (data != null && data.Exists(str => str.Equals(name)))
                        return data[parameters.FindIndex(str => str.Equals(guid))];
                }
            }
            return String.Empty;
        }

        /// <summary>
        /// Обязательный метод плагина импорта. Получает графические параметры фигур принадлежащие определённому мастеру.
        /// </summary>
        /// <param name="className">Имя мастера. Получаем заголовки параметров у объектов данного мастера</param>
        /// <param name="partNumber">Номер текущего информационного пакета</param>
        /// <returns>Возвращает параметры фигур принадлежащие определённому мастеру</returns>
        public static List<string> GetGraphicParameters(string className, int partNumber)
        {
            try
            {
                if (GraphicParameterTable.Count > 0 && GraphicParameterTable.ContainsKey(className))
                {
                    return (List<string>)GraphicParameterTable[className];
                }
                List<string> parameters = new List<string>();

                if (className.Equals(Diagram))
                {
                    AddWithCheck(parameters, Shape_guid);
                    AddWithCheck(parameters, PageWidth);
                    AddWithCheck(parameters, PageHeight);
                }
                else
                {
                    AddWithCheck(parameters, Master);
                    AddWithCheck(parameters, ParentGuid);
                    AddWithCheck(parameters, Shape_guid);
                    AddWithCheck(parameters, Text);
                    AddWithCheck(parameters, Width);
                    AddWithCheck(parameters, Height);

                    AddWithCheck(parameters, FontSize);

                    AddWithCheck(parameters, XLeft);
                    AddWithCheck(parameters, YTop);
                    AddWithCheck(parameters, XRight);
                    AddWithCheck(parameters, YBottom);

                    AddWithCheck(parameters, XCenter);
                    AddWithCheck(parameters, YCenter);

                    AddWithCheck(parameters, BeginX);
                    AddWithCheck(parameters, BeginY);
                    AddWithCheck(parameters, EndX);
                    AddWithCheck(parameters, EndY);

                    AddWithCheck(parameters, ShapeFrom_guid);
                    AddWithCheck(parameters, ShapeTo_guid);

                    AddWithCheck(parameters, Geometry);

                    AddWithCheck(parameters, PinConnectionGuid);
                }

                parameters.AddRange(GetParameters(className, partNumber));

                if (!GraphicParameterTable.ContainsKey(className))
                    GraphicParameterTable.Add(className, parameters);

                return parameters;
            }
            catch (Exception ex)
            {
                throw new Exception(ex.Message, ex.InnerException);
            }
        }

        /// </summary>
        /// Обязательный метод плагина импорта. Получает значения графических параметров фигур определённого мастера.
        /// <param name="className">Имя мастера</param>
        /// <param name="partNumber">Номер текущего информационного пакета</param>
        /// <returns>Возвращает коллекцию содержащую список массивов со значениями параметров</returns>
        public static List<string[]> GetGraphicData(string className, int partNumber)
        {
            try
            {
                List<string> parameters = new List<string>();
                if (GraphicParameterTable.Count > 0 && GraphicParameterTable.ContainsKey(className))
                    parameters = (List<string>)GraphicParameterTable[className];
                else
                    parameters = GetGraphicParameters(className, partNumber);

                List<string[]> master = new List<string[]>();

                Document doc = GetDocumentByPartNumber(partNumber);
                foreach (Page page in doc.Pages)
                {
                    if (className.Equals(Diagram))
                    {
                        master.Add(new string[parameters.Count]);
                        //Обрабатываем диаграмму отдельно. Для графического импорта требуются только глобальный идентификатор, ширина и высота страницы.
                        
                        master[master.Count - 1][FindArrayIndex(parameters, Shape_guid)] = GetShapeGUID(page.PageSheet);//Добавляем гуид, по нему производится поиск объектов в БС
                        master[master.Count - 1][FindArrayIndex(parameters, PageWidth)] = page.PageSheet.get_CellsSRC((short)VisSectionIndices.visSectionObject, (short)VisRowIndices.visRowPage, (short)VisCellIndices.visPageWidth).FormulaU;
                        master[master.Count - 1][FindArrayIndex(parameters, PageHeight)] = page.PageSheet.get_CellsSRC((short)VisSectionIndices.visSectionObject, (short)VisRowIndices.visRowPage, (short)VisCellIndices.visPageHeight).FormulaU;

                        continue;
                    }

                    foreach (Shape shape in page.Shapes)
                        if (className.Equals(GetMasterName(shape, true)))
                            GetGeometryList(master, shape, parameters);
                }

                List<string[]> shapeData = GetData(className, partNumber);

                for (int i = 0; i < master.Count; i++)
                {
                    int indent = master[i].Length - shapeData[i].Length;
                    for (int j = indent; j < master[i].Length; j++)
                    {
                        master[i][j] = shapeData[i][j - indent];
                    }
                }

                return master;
            }
            catch (Exception ex)
            {
                throw new Exception(ex.Message, ex.InnerException);
            }
        }

        /// <summary>
        /// Метод предназначен для получения геометрии из фигуры и записи её в список.
        /// </summary>
        /// <param name="masterString">List, куда записываем геометрию</param>
        /// <param name="shape">Фигура, у которого считываем геометрию</param>
        private static void GetGeometryList(List<string[]> masterString, Shape shape, List<string> parameters)
        {
            string shapeGuid = GetShapeGUID(shape);
            if (GraphicShapeDataList.Count > 0 && GraphicShapeDataList.ContainsKey(shapeGuid))
            {
                masterString.Add((string[])GraphicShapeDataList[shapeGuid]);
                return;
            }

            ///Числовые размерности берутся по умолчанию(дюймы). 
            ///Если размерность не дюймы, требуется точно указать размерность с помощью таблицы Visio https://msdn.microsoft.com/en-us/library/office/ff767977.aspx

            string[] geometry = new string[parameters.Count];
            geometry[FindArrayIndex(parameters, Master)] = GetMasterName(shape);//Добавление здесь мастера связано с общей логикой плагинов, по этому мастеру будет создан шейп

            geometry[FindArrayIndex(parameters, ParentGuid)] = GetShapeGUID(shape.ContainingPage.PageSheet);
            geometry[FindArrayIndex(parameters, Shape_guid)] = shapeGuid;//Добавляем глобальный идентификатор, по нему производится поиск объектов в BS
            geometry[FindArrayIndex(parameters, Text)] = GetShapeText(shape, true);
            geometry[FindArrayIndex(parameters, Width)] = shape.Cells[Width].ResultIU.ToString();
            geometry[FindArrayIndex(parameters, Height)] = shape.Cells[Height].ResultIU.ToString();
            bool oneD = IsOneD(shape);

            geometry[FindArrayIndex(parameters, FontSize)] = GetCellValue(shape, (short)VisSectionIndices.visSectionCharacter, (short)VisRowIndices.visRowCharacter, (short)VisCellIndices.visCharacterSize);

            VisBoundingBoxArgs boundingBoxArgs = oneD ? VisBoundingBoxArgs.visBBoxUprightText : VisBoundingBoxArgs.visBBoxExtents;

            double x1, y1, x2, y2;
            shape.BoundingBox((short)boundingBoxArgs, out x1, out y1, out x2, out y2);

            geometry[FindArrayIndex(parameters, XLeft)] = x1.ToString();
            geometry[FindArrayIndex(parameters, YTop)] = y2.ToString();
            geometry[FindArrayIndex(parameters, XRight)] = x2.ToString();
            geometry[FindArrayIndex(parameters, YBottom)] = y1.ToString();

            geometry[FindArrayIndex(parameters, XCenter)] = (Math.Abs(x1 + x2) / 2).ToString();
            geometry[FindArrayIndex(parameters, YCenter)] = (Math.Abs(y1 + y2) / 2).ToString();

            if (oneD)//Для стрелок особенная логика
            {
                short section = (short)VisSectionIndices.visSectionObject;
                short row = (short)VisRowIndices.visRowXForm1D;

                geometry[FindArrayIndex(parameters, BeginX)] = GetCellValue(shape, section, row, (short)VisCellIndices.vis1DBeginX, VisUnitCodes.visInches);
                geometry[FindArrayIndex(parameters, BeginY)] = GetCellValue(shape, section, row, (short)VisCellIndices.vis1DBeginY, VisUnitCodes.visInches);
                geometry[FindArrayIndex(parameters, EndX)] = GetCellValue(shape, section, row, (short)VisCellIndices.vis1DEndX, VisUnitCodes.visInches);
                geometry[FindArrayIndex(parameters, EndY)] = GetCellValue(shape, section, row, (short)VisCellIndices.vis1DEndY, VisUnitCodes.visInches);

                //Забираем guid объектов из которого выходит стрелка и в какой входит
                string[] connects = GetGuidIn_Out(shape);
                geometry[FindArrayIndex(parameters, ShapeFrom_guid)] = connects[0];
                geometry[FindArrayIndex(parameters, ShapeTo_guid)] = connects[1];

                //Здесь же забираем сложную геометрию стрелки
                section = (short)VisSectionIndices.visSectionFirstComponent;
                row = (short)VisRowIndices.visRowFirst;

                string coord = GetGeometryCoords(shape, section, row);

                geometry[FindArrayIndex(parameters, Geometry)] = String.IsNullOrEmpty(coord) ? coord : coord.Substring(0, coord.Length - 1);
            }
            else
            {
                string[] connects = GetGuidIn_Out(shape);
                geometry[FindArrayIndex(parameters, PinConnectionGuid)] = connects[2];
            }
            masterString.Add(geometry);

            if(!GraphicShapeDataList.ContainsKey(shapeGuid))
                GraphicShapeDataList.Add(shapeGuid, masterString[masterString.Count - 1]);
        }

        /// <summary>
        /// Получает координаты геометрии стрелок.
        /// </summary>
        /// <param name="shape">Фигура-стрелка у которой забираем координаты</param>
        /// <param name="section">Константа(short) отвечающая за определённую секцию</param>
        /// <param name="row">Константа(short) отвечающая за определённую строку</param>
        /// <returns></returns>
        private static string GetGeometryCoords(Shape shape, short section, short row)
        {
            short cellX = (short)VisCellIndices.visX;
            short cellY = (short)VisCellIndices.visY;
            string coord = String.Empty;
            short moveToTag = (short)VisRowTags.visTagMoveTo;
            //Забираем значения координат почерёдно из всех секций геометрии, из всех строк в каждой секции.
            while (shape.get_SectionExists(section, 0) != 0)
            {
                while (CellExist(shape, section, row, cellX))
                {
                    short rowType = shape.get_RowType(section, row);
                    if (rowType <= (short)VisRowTags.visTagArcTo && rowType >= moveToTag)
                    {
                        if (rowType == moveToTag)
                            if (String.IsNullOrEmpty(coord))
                            {
                                coord += (GetCellValue(shape, section, row, cellX, VisUnitCodes.visInches)) + ";";
                                coord += (GetCellValue(shape, section, row, cellY, VisUnitCodes.visInches)) + ";";
                            }
                            //Пропускаем строку, если внутри геометрии есть ещё один MoveTo
                        
                        if (rowType == (short)VisRowTags.visTagLineTo)
                        {
                            coord += (GetCellValue(shape, section, row, cellX, VisUnitCodes.visInches)) + ";";
                            coord += (GetCellValue(shape, section, row, cellY, VisUnitCodes.visInches)) + ";";
                        }
                        if (rowType == (short)VisRowTags.visTagArcTo && !String.IsNullOrEmpty(coord))
                        {
                            coord = coord.Substring(0, coord.Length - 1);
                            coord = coord.Substring(0, coord.LastIndexOf(';'));
                            coord = coord.Substring(0, coord.LastIndexOf(';') + 1);
                        }
                    }
                    row++;
                }
                row = 1;
                section++;
            }
            //Код ниже предназначен только для сложных стрелок, учитывает то, что координаты лежат в подфигурах.
            if (String.IsNullOrEmpty(coord) && shape.Shapes != null)
            {
                section = (short)VisSectionIndices.visSectionFirstComponent;
                row = (short)VisRowIndices.visRowFirst;
                foreach (Shape subShape in shape.Shapes)
                {
                    if (!IsOneD(subShape)) continue;

                    coord = GetGeometryCoords(subShape, section, row);

                    break;
                }
            }
            return coord;
        }

        /// <summary>
        /// Получает массив глобальных идентификаторов, соответствующих фигурам присоединённым к началу и к концу стрелки.
        /// </summary>
        /// <param name="shape">Стрелка, у которой ищем присоединённые фигуры</param>
        /// <param name="geometry">Список строк - параметров фигуры</param>
        /// <param name="paramNames">Список строк - названий параметров фигуры</param>
        private static string[] GetGuidIn_Out(Shape shape)
        {
            String[] connects = new String[3] { String.Empty, String.Empty, String.Empty };
            
            if (shape.Connects != null && shape.Connects.Count > 0)
                foreach (Connect connect in shape.Connects)
                {
                    if (connect.FromPart.Equals((short)VisFromParts.visBegin))
                        connects[0] = GetShapeGUID(GetParentShape(connect.ToSheet));
                    else if (connect.FromPart.Equals((short)VisFromParts.visEnd))
                        connects[1] = GetShapeGUID(GetParentShape(connect.ToSheet));
                    else if (connect.FromPart.Equals((short)VisFromParts.visFromPin))
                    {
                        //Забираем guid фигуры соединённой с текущей фигурой в точке
                        connects[2] = GetShapeGUID(GetParentShape(connect.ToSheet));
                    }
                }
            
            return connects;
        }

        /// <summary>
        /// Метод получает родительскую фигуру.
        /// </summary>
        /// <param name="shape">Шейп, родителя которого ищем</param>
        /// <returns>Шейп родитель</returns>
        private static Shape GetParentShape(Shape shape)
        {
            while (shape.ContainingShape != null && shape.ContainingShape.Type != (short)VisShapeTypes.visTypePage)
                shape = shape.ContainingShape;

            return shape;
        }
        
        /// <summary>
        /// Получает уникальный глобальный идентификатор фигуры.
        /// </summary>
        /// <param name="shape">Фигура, от которой получаем идентификатор</param>
        /// <returns>Возвращает уникальный глобальный идентификатор фигуры</returns>
        private static string GetShapeGUID(Shape shape)
        {
            return shape.get_UniqueID((short)VisUniqueIDArgs.visGetOrMakeGUID).Replace("{", String.Empty).Replace("}", String.Empty);
        }

        /// <summary>
        /// Получает текст фигуры с учётом некоторых особенностей.
        /// </summary>
        /// <param name="shape">Фигура, из которой получаем текст</param>
        /// <returns>Текст фигуры</returns>
        private static string GetShapeText(Shape shape, bool isGraphicText = false)
        {
            string text = String.Empty;
            short cellIndices = (short)VisCellIndices.visFieldCell;
            short sectionIndices = (short)VisSectionIndices.visSectionTextField;

            //Если у объекта есть секция Text Fields, забираем текст из неё.
            if (shape.get_SectionExists(sectionIndices, 0) != 0)
                text = GetCellValue(shape, sectionIndices, 0, cellIndices);
            if (String.IsNullOrEmpty(text))
            {
                if (!String.IsNullOrEmpty(shape.Text) && !shape.Text.Equals(new string(new char[] { UnicodeObject })))
                    text = shape.Text;
                else
                {
                    if (shape.Shapes != null)
                        foreach (Shape subShape in shape.Shapes) //Ищем нижний элемент группы, позволяющий редактироваться
                        {
                            Cell textCell = subShape.get_CellsSRC(
                                (short)VisSectionIndices.visSectionObject,
                                (short)VisRowIndices.visRowLock,
                                (short)VisCellIndices.visLockTextEdit
                            );
                            if (textCell.FormulaU == "1")//Проверка на возможность редактирования текста фигуры.
                                continue;

                            if (shape.get_SectionExists(sectionIndices, 0) != 0)
                            {
                                text = GetCellValue(subShape, sectionIndices, 0, cellIndices);
                                break;
                            }

                            if (!String.IsNullOrEmpty(subShape.Text) && !subShape.Text.Equals(new string(new char[] { UnicodeObject })))
                            {
                                text = subShape.Text;
                                break;
                            }
                        }
                }
            }
            return//Объектный текст не должен содержать переносов и лишних символов
                isGraphicText ?
                    text
                    :
                    text.Replace("\r", "").Trim('\n').Replace("\n", " ").Replace("  ", " ");
        }

        /// <summary>
        /// Освобождает память, занятую передеваемой хеш таблицей.
        /// </summary>
        /// <param name="table">Хеш таблица, которую надо обнулить</param>
        private static void ClearHashTable(Hashtable table)
        {
            if (table != null)
                table = null;
        }

        /// <summary>
        /// Метод используется для освобождения ресурсов после окончания работы с плагином.
        /// </summary>
        public static void Close()
        {
            try
            {
                if (FilesPath != null)
                    FilesPath = null;
                
                ClearHashTable(MastersTable);
                ClearHashTable(GraphicParameterTable);
                ClearHashTable(ObjectParameterTable);
                ClearHashTable(GraphicShapeDataList);
                ClearHashTable(ShapeDataList);

                if (VisioApplication != null)
                    VisioApplication.Quit();
            }
            catch (Exception ex)
            {
                throw new Exception(ex.Message, ex.InnerException);
            }
        }
    }
}
